<?php

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Cart Blocker
 * 
 * Bloqueia o carrinho do WooCommerce e força o uso dos popups do plugin
 * para garantir que os clientes sempre passem pelo fluxo correto
 */

class Upgram_Cart_Blocker
{
    private static $instance = null;

    public static function get_instance()
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        // Hook para bloquear acesso direto ao carrinho - usando init para garantir que funcione
        add_action('init', array($this, 'block_cart_access'));
        
        // Hook para bloquear checkout direto
        add_action('init', array($this, 'block_direct_checkout'));
        
        // Hook para remover botões de checkout do carrinho
        add_action('woocommerce_cart_actions', array($this, 'remove_checkout_buttons'));
        
        // Hook para adicionar popup automático no carrinho
        add_action('wp_footer', array($this, 'add_cart_popup'));
        
        // Hook para bloquear adição direta ao carrinho
        add_action('woocommerce_add_to_cart_validation', array($this, 'block_direct_add_to_cart'), 10, 3);
        
        // Hook para interceptar redirecionamentos para carrinho
        add_filter('woocommerce_add_to_cart_redirect', array($this, 'redirect_to_popup_instead_of_cart'));
        
        // Hook para bloquear mini-cart
        add_action('wp_footer', array($this, 'block_mini_cart'));
        
        // Hook para AJAX - obter dados do produto para popup
        add_action('wp_ajax_upgram_get_product_for_popup', array($this, 'get_product_for_popup'));
        add_action('wp_ajax_nopriv_upgram_get_product_for_popup', array($this, 'get_product_for_popup'));
        
        // Hook adicional para interceptar antes da página carregar
        add_action('wp', array($this, 'intercept_page_access'));
        
        // Hook com prioridade alta para garantir que funcione
        add_action('template_redirect', array($this, 'force_block_pages'), 1);
    }

    /**
     * Força bloqueio das páginas com prioridade alta
     */
    public function force_block_pages()
    {
        // Log para debug - verificar se a função está sendo chamada
        error_log('Upgram Cart Blocker: force_block_pages() chamada');
        
        // Verificar se o bloqueio está ativado (padrão é true se não configurado)
        $block_cart = get_option('upgram_block_cart');
        if ($block_cart === false) {
            error_log('Upgram Cart Blocker: Bloqueio explicitamente desabilitado');
            return; // Explicitamente desabilitado
        }
        // Se não configurado (null) ou true, ativa o bloqueio
        error_log('Upgram Cart Blocker: Bloqueio ativado - verificando páginas');
        
        // Verificar se está na página do carrinho
        if (is_cart()) {
            // Log para debug
            error_log('Upgram Cart Blocker: Carrinho acessado - bloqueando');
            // Sempre bloquear acesso ao carrinho
            $this->show_cart_blocked_message();
            exit;
        }
        
        // Verificar se está na página de checkout
        if (is_checkout()) {
            // Verificar se o usuário veio do popup do plugin
            $from_popup = isset($_GET['from_upgram_popup']) && $_GET['from_upgram_popup'] === '1';
            
            if (!$from_popup) {
                // Log para debug
                error_log('Upgram Cart Blocker: Checkout acessado sem popup - bloqueando');
                // Não veio do popup - bloquear acesso
                $this->show_checkout_blocked_message();
                exit;
            } else {
                // Log para debug
                error_log('Upgram Cart Blocker: Checkout acessado via popup - permitindo');
            }
        }
    }

    /**
     * Intercepta acesso às páginas antes de carregar
     */
    public function intercept_page_access()
    {
        // Verificar se o bloqueio está ativado
        if (!get_option('upgram_block_cart', true)) {
            return;
        }
        
        // Verificar se está na página do carrinho
        if (is_cart()) {
            // Verificar se há produtos no carrinho
            if (WC()->cart->is_empty()) {
                // Carrinho vazio - redirecionar para home
                wp_redirect(home_url());
                exit;
            }
        }
        
        // Verificar se está na página de checkout
        if (is_checkout()) {
            // Verificar se o usuário veio do popup do plugin
            $from_popup = isset($_GET['from_upgram_popup']) && $_GET['from_upgram_popup'] === '1';
            
            if (!$from_popup) {
                // Não veio do popup - redirecionar para home
                wp_redirect(home_url());
                exit;
            }
        }
    }

    /**
     * Bloqueia acesso direto ao carrinho
     */
    public function block_cart_access()
    {
        // Verificar se o bloqueio está ativado
        if (!get_option('upgram_block_cart', true)) {
            return;
        }
        
        if (is_cart()) {
            // Verificar se há produtos no carrinho
            if (WC()->cart->is_empty()) {
                // Carrinho vazio - redirecionar para home
                wp_redirect(home_url());
                exit;
            }
        }
    }

    /**
     * Bloqueia checkout direto
     */
    public function block_direct_checkout()
    {
        // Verificar se o bloqueio está ativado
        if (!get_option('upgram_block_cart', true)) {
            return;
        }
        
        if (is_checkout()) {
            // Verificar se o usuário veio do popup do plugin
            $from_popup = isset($_GET['from_upgram_popup']) && $_GET['from_upgram_popup'] === '1';
            
            if (!$from_popup) {
                // Não veio do popup - redirecionar para carrinho (que mostrará o popup)
                wp_redirect(wc_get_cart_url());
                exit;
            }
        }
    }

    /**
     * Remove botões de checkout do carrinho
     */
    public function remove_checkout_buttons()
    {
        // Verificar se o bloqueio está ativado
        if (!get_option('upgram_block_cart', true)) {
            return;
        }
        
        // Remover botão "Finalizar Compra"
        remove_action('woocommerce_cart_actions', 'woocommerce_button_proceed_to_checkout', 20);
        
        // Adicionar mensagem explicativa
        echo '<div class="upgram-cart-blocked-notice" style="background: #fff3cd; border: 1px solid #ffeaa7; color: #856404; padding: 15px; margin: 20px 0; border-radius: 4px;">';
        echo '<h3 style="margin: 0 0 10px 0; color: #856404;">🚫 Compra Bloqueada</h3>';
        echo '<p style="margin: 0;">Para finalizar sua compra, você deve usar o popup de compra do plugin. Clique no botão abaixo para abrir o popup correto.</p>';
        echo '<button type="button" class="button button-primary" onclick="upgramShowCartPopup()" style="margin-top: 10px;">Abrir Popup de Compra</button>';
        echo '</div>';
    }

    /**
     * Adiciona popup automático no carrinho
     */
    public function add_cart_popup()
    {
        // Verificar se o bloqueio e popup automático estão ativados
        if (!get_option('upgram_block_cart', true) || !get_option('upgram_auto_show_popup', true)) {
            return;
        }
        
        if (is_cart() && !WC()->cart->is_empty()) {
            $this->show_cart_popup();
        }
    }

    /**
     * Mostra popup do carrinho
     */
    private function show_cart_popup()
    {
        $cart_items = WC()->cart->get_cart();
        $first_item = reset($cart_items);
        
        if ($first_item) {
            $product_id = $first_item['product_id'];
            $variation_id = $first_item['variation_id'];
            
            // Obter dados do produto
            $product = wc_get_product($variation_id ?: $product_id);
            
            if ($product) {
                ?>
                <script type="text/javascript">
                jQuery(document).ready(function($) {
                    // Mostrar popup automaticamente
                    upgramShowCartPopup();
                });
                
                function upgramShowCartPopup() {
                    // Obter dados do produto via AJAX
                    jQuery.post(ajaxurl, {
                        action: 'upgram_get_product_for_popup',
                        product_id: <?php echo $product_id; ?>,
                        variation_id: <?php echo $variation_id ?: 'null'; ?>,
                        security: '<?php echo wp_create_nonce('upgram_get_product_for_popup'); ?>'
                    }, function(response) {
                        if (response.success) {
                            // Inserir HTML do popup
                            $('body').append(response.data.html);
                            
                            // Mostrar popup
                            $('#upgram-initial-modal').show();
                            
                            // Esconder carrinho
                            $('.woocommerce-cart-form').hide();
                            $('.cart-collaterals').hide();
                            
                            // Adicionar mensagem
                            $('.woocommerce-cart-form').before('<div class="upgram-cart-redirect-notice" style="background: #d1ecf1; border: 1px solid #bee5eb; color: #0c5460; padding: 15px; margin: 20px 0; border-radius: 4px;"><h3 style="margin: 0 0 10px 0;">🛒 Redirecionado para Popup</h3><p style="margin: 0;">Para finalizar sua compra, use o popup abaixo. O carrinho foi bloqueado para garantir o fluxo correto.</p></div>');
                        }
                    });
                }
                </script>
                <?php
            }
        }
    }

    /**
     * Bloqueia adição direta ao carrinho
     */
    public function block_direct_add_to_cart($valid, $product_id, $quantity)
    {
        // Verificar se o bloqueio está ativado
        if (!get_option('upgram_block_cart', true)) {
            return $valid;
        }
        
        // Verificar se a requisição veio de um popup do plugin
        $from_popup = isset($_POST['from_upgram_popup']) && $_POST['from_upgram_popup'] === '1';
        
        if (!$from_popup) {
            // Não veio do popup - bloquear
            wc_add_notice('Você deve usar o popup de compra para adicionar produtos ao carrinho.', 'error');
            return false;
        }
        
        return $valid;
    }

    /**
     * Redireciona para popup em vez do carrinho
     */
    public function redirect_to_popup_instead_of_cart($url)
    {
        // Verificar se o bloqueio está ativado
        if (!get_option('upgram_block_cart', true)) {
            return $url;
        }
        
        // Se não veio do popup, redirecionar para a página do produto
        $from_popup = isset($_POST['from_upgram_popup']) && $_POST['from_upgram_popup'] === '1';
        
        if (!$from_popup) {
            $product_id = isset($_POST['add-to-cart']) ? intval($_POST['add-to-cart']) : 0;
            if ($product_id) {
                return get_permalink($product_id);
            }
        }
        
        return $url;
    }

    /**
     * Bloqueia mini-cart
     */
    public function block_mini_cart()
    {
        // Verificar se o bloqueio está ativado
        if (!get_option('upgram_block_cart', true)) {
            return;
        }
        
        ?>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            // Remover mini-cart
            $('.woocommerce-cart-fragments').remove();
            $('.cart-contents').remove();
            $('.cart-fragments').remove();
            
            // Interceptar cliques em links do carrinho
            $(document).on('click', 'a[href*="cart"]', function(e) {
                e.preventDefault();
                alert('Para acessar o carrinho, use o popup de compra do plugin.');
                return false;
            });
        });
        </script>
        <?php
    }

    /**
     * Mostra mensagem de carrinho bloqueado
     */
    private function show_cart_blocked_message()
    {
        ?>
        <!DOCTYPE html>
        <html <?php language_attributes(); ?>>
        <head>
            <meta charset="<?php bloginfo('charset'); ?>">
            <meta name="viewport" content="width=device-width, initial-scale=1">
            <title>Carrinho Bloqueado - <?php bloginfo('name'); ?></title>
            <style>
                body {
                    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                    margin: 0;
                    padding: 0;
                    min-height: 100vh;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                }
                .blocked-container {
                    background: white;
                    padding: 40px;
                    border-radius: 12px;
                    box-shadow: 0 20px 40px rgba(0,0,0,0.1);
                    text-align: center;
                    max-width: 500px;
                    margin: 20px;
                }
                .blocked-icon {
                    font-size: 64px;
                    margin-bottom: 20px;
                }
                .blocked-title {
                    font-size: 24px;
                    font-weight: 600;
                    color: #e74c3c;
                    margin-bottom: 15px;
                }
                .blocked-message {
                    color: #666;
                    font-size: 16px;
                    line-height: 1.6;
                    margin-bottom: 30px;
                }
                .blocked-button {
                    background: #0073aa;
                    color: white;
                    padding: 12px 24px;
                    border: none;
                    border-radius: 6px;
                    font-size: 16px;
                    cursor: pointer;
                    text-decoration: none;
                    display: inline-block;
                    transition: background 0.3s;
                }
                .blocked-button:hover {
                    background: #005a87;
                }
                .blocked-info {
                    background: #f8f9fa;
                    padding: 20px;
                    border-radius: 8px;
                    margin-top: 20px;
                    border-left: 4px solid #0073aa;
                }
                .blocked-info h4 {
                    margin: 0 0 10px 0;
                    color: #0073aa;
                }
                .blocked-info p {
                    margin: 0;
                    color: #666;
                    font-size: 14px;
                }
            </style>
        </head>
        <body>
            <div class="blocked-container">
                <div class="blocked-icon">🛒</div>
                <h1 class="blocked-title">Carrinho Bloqueado</h1>
                <p class="blocked-message">
                    O acesso direto ao carrinho foi bloqueado para garantir que você use o popup de compra correto.
                </p>
                <a href="<?php echo home_url(); ?>" class="blocked-button">
                    Voltar ao Início
                </a>
                <div class="blocked-info">
                    <h4>ℹ️ Por que isso acontece?</h4>
                    <p>Para finalizar sua compra corretamente, você deve usar o popup de compra do plugin. Isso garante que todos os dados necessários sejam coletados.</p>
                </div>
            </div>
        </body>
        </html>
        <?php
    }

    /**
     * Mostra mensagem de checkout bloqueado
     */
    private function show_checkout_blocked_message()
    {
        ?>
        <!DOCTYPE html>
        <html <?php language_attributes(); ?>>
        <head>
            <meta charset="<?php bloginfo('charset'); ?>">
            <meta name="viewport" content="width=device-width, initial-scale=1">
            <title>Checkout Bloqueado - <?php bloginfo('name'); ?></title>
            <style>
                body {
                    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                    margin: 0;
                    padding: 0;
                    min-height: 100vh;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                }
                .blocked-container {
                    background: white;
                    padding: 40px;
                    border-radius: 12px;
                    box-shadow: 0 20px 40px rgba(0,0,0,0.1);
                    text-align: center;
                    max-width: 500px;
                    margin: 20px;
                }
                .blocked-icon {
                    font-size: 64px;
                    margin-bottom: 20px;
                }
                .blocked-title {
                    font-size: 24px;
                    font-weight: 600;
                    color: #e74c3c;
                    margin-bottom: 15px;
                }
                .blocked-message {
                    color: #666;
                    font-size: 16px;
                    line-height: 1.6;
                    margin-bottom: 30px;
                }
                .blocked-button {
                    background: #0073aa;
                    color: white;
                    padding: 12px 24px;
                    border: none;
                    border-radius: 6px;
                    font-size: 16px;
                    cursor: pointer;
                    text-decoration: none;
                    display: inline-block;
                    transition: background 0.3s;
                }
                .blocked-button:hover {
                    background: #005a87;
                }
                .blocked-info {
                    background: #f8f9fa;
                    padding: 20px;
                    border-radius: 8px;
                    margin-top: 20px;
                    border-left: 4px solid #0073aa;
                }
                .blocked-info h4 {
                    margin: 0 0 10px 0;
                    color: #0073aa;
                }
                .blocked-info p {
                    margin: 0;
                    color: #666;
                    font-size: 14px;
                }
            </style>
        </head>
        <body>
            <div class="blocked-container">
                <div class="blocked-icon">🚫</div>
                <h1 class="blocked-title">Checkout Bloqueado</h1>
                <p class="blocked-message">
                    O acesso direto ao checkout foi bloqueado. Você deve usar o popup de compra para finalizar sua compra.
                </p>
                <a href="<?php echo home_url(); ?>" class="blocked-button">
                    Voltar ao Início
                </a>
                <div class="blocked-info">
                    <h4>ℹ️ Como proceder?</h4>
                    <p>Para finalizar sua compra, acesse um produto e use o popup de compra. Isso garante que todos os dados necessários sejam coletados corretamente.</p>
                </div>
            </div>
        </body>
        </html>
        <?php
    }

    /**
     * AJAX - Obter dados do produto para popup
     */
    public function get_product_for_popup()
    {
        check_ajax_referer('upgram_get_product_for_popup', 'security');
        
        $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
        $variation_id = isset($_POST['variation_id']) ? intval($_POST['variation_id']) : 0;
        
        if (!$product_id) {
            wp_send_json_error(array('message' => 'Produto inválido'));
        }
        
        $product = wc_get_product($variation_id ?: $product_id);
        
        if (!$product) {
            wp_send_json_error(array('message' => 'Produto não encontrado'));
        }
        
        // Obter dados do produto
        $product_data = array(
            'id' => $product_id,
            'variation_id' => $variation_id,
            'name' => $product->get_name(),
            'price' => $product->get_price(),
            'description' => $product->get_short_description(),
            'image' => wp_get_attachment_image_url($product->get_image_id(), 'medium')
        );
        
        // Gerar HTML do popup
        ob_start();
        $this->render_popup_html($product_data);
        $html = ob_get_clean();
        
        wp_send_json_success(array('html' => $html));
    }

    /**
     * Renderiza HTML do popup
     */
    private function render_popup_html($product_data)
    {
        ?>
        <div id="upgram-initial-modal" class="upgram-modal" style="display: none;">
            <div class="upgram-modal-content">
                <div class="upgram-modal-header">
                    <h2>🛒 Finalizar Compra</h2>
                    <span class="upgram-modal-close" onclick="upgramCloseModal()">&times;</span>
                </div>
                <div class="upgram-modal-body">
                    <div class="upgram-product-info">
                        <h3><?php echo esc_html($product_data['name']); ?></h3>
                        <p class="upgram-price">R$ <?php echo number_format($product_data['price'], 2, ',', '.'); ?></p>
                        <?php if ($product_data['description']): ?>
                            <p class="upgram-description"><?php echo esc_html($product_data['description']); ?></p>
                        <?php endif; ?>
                    </div>
                    
                    <div class="upgram-cart-notice">
                        <p><strong>⚠️ Atenção:</strong> Para finalizar sua compra, você deve usar o popup de compra do plugin. O carrinho tradicional foi bloqueado para garantir o fluxo correto.</p>
                    </div>
                    
                    <div class="upgram-modal-actions">
                        <button type="button" class="button button-primary" onclick="upgramStartPurchase()">
                            Iniciar Compra no Popup
                        </button>
                        <button type="button" class="button button-secondary" onclick="upgramCloseModal()">
                            Cancelar
                        </button>
                    </div>
                </div>
            </div>
        </div>
        
        <script type="text/javascript">
        function upgramStartPurchase() {
            // Redirecionar para página do produto com parâmetro para abrir popup
            const productUrl = '<?php echo get_permalink($product_data['id']); ?>?variation_id=<?php echo $product_data['variation_id']; ?>&open_popup=1';
            window.location.href = productUrl;
        }
        
        function upgramCloseModal() {
            $('#upgram-initial-modal').hide();
            // Redirecionar para home se fechar
            window.location.href = '<?php echo home_url(); ?>';
        }
        </script>
        
        <style>
        .upgram-modal {
            position: fixed;
            z-index: 9999;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.5);
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .upgram-modal-content {
            background-color: #fff;
            padding: 0;
            border-radius: 8px;
            max-width: 500px;
            width: 90%;
            max-height: 80vh;
            overflow-y: auto;
            box-shadow: 0 4px 20px rgba(0,0,0,0.3);
        }
        
        .upgram-modal-header {
            background: #0073aa;
            color: white;
            padding: 20px;
            border-radius: 8px 8px 0 0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .upgram-modal-header h2 {
            margin: 0;
            font-size: 18px;
        }
        
        .upgram-modal-close {
            font-size: 24px;
            cursor: pointer;
            line-height: 1;
        }
        
        .upgram-modal-body {
            padding: 20px;
        }
        
        .upgram-product-info {
            margin-bottom: 20px;
            padding: 15px;
            background: #f9f9f9;
            border-radius: 4px;
        }
        
        .upgram-product-info h3 {
            margin: 0 0 10px 0;
            color: #333;
        }
        
        .upgram-price {
            font-size: 18px;
            font-weight: bold;
            color: #0073aa;
            margin: 0 0 10px 0;
        }
        
        .upgram-description {
            margin: 0;
            color: #666;
            font-size: 14px;
        }
        
        .upgram-cart-notice {
            background: #fff3cd;
            border: 1px solid #ffeaa7;
            color: #856404;
            padding: 15px;
            border-radius: 4px;
            margin-bottom: 20px;
        }
        
        .upgram-modal-actions {
            display: flex;
            gap: 10px;
            justify-content: flex-end;
        }
        
        .upgram-modal-actions .button {
            padding: 10px 20px;
            border-radius: 4px;
            text-decoration: none;
            border: none;
            cursor: pointer;
        }
        
        .upgram-modal-actions .button-primary {
            background: #0073aa;
            color: white;
        }
        
        .upgram-modal-actions .button-secondary {
            background: #f1f1f1;
            color: #333;
        }
        </style>
        <?php
    }
}

// Inicializar o bloqueador de carrinho
Upgram_Cart_Blocker::get_instance();
